/* ////////////////////////////////////////////////////////////////////////////////////////////////
Name: 			import_new_leagues.do
Description: 	Imports the text files, cleans them, and saves the data as a dta

Notes: 			- Created by Jeff 
				- Last updated 8/24/2022
//////////////////////////////////////////////////////////////////////////////////////////////// */

* Set up environment ------------------------------------------------------------------------------
clear

set scheme plotplainblind

*set processors 24
*set max_memory 115g

*Add in your file path with cap in front
cap cd "C:/Users/jcross/Dropbox/HFA and VAR"

* Raw data
global rawdata "1_data/0_raw"
global cleandata "1_data/1_clean"

* Results
global graphs "4_results/Figures"
global regressions "4_results/Tables"


************************ Importing First ************************************
	local leagues "australia denmark dutch france greece portugal russia scotland switzerland turkey"
	local leagues_inv "brazil japan korea"

	tempfile base
	save `base', emptyok

	foreach l of local leagues {
		forvalues i = 10(1)18 {
			clear
			local j = `i'+1	
			cap import delimited "${rawdata}/`l'_`i'_`j'.txt", stringcols(_all)
			gen country = "`l'"
			gen season = "`i'`j'"
			cap assert _N == 0
			if _rc != 0 {
				append using `base'
				save `base', replace
			}		
		}
	}

	foreach l of local leagues_inv {
		forvalues i = 14(1)19 {
			clear
			cap import delimited "${rawdata}/`l'_`i'.txt", stringcols(_all)
			gen country = "`l'"
			gen season = "`i'"
			cap assert _N == 0
			if _rc != 0 {
				append using `base'
				save `base', replace
			}		
		}		
	}

	use `base', clear

************************ Cleaning now ************************************
	drop if wk == "" // these are missing observations or playoffs/relegation matches
	drop if round == "Championship round" | round == "European competition play-offs" | round == "Relegation round"
	drop matchreport
	
	***** Cleaning or creating new variables *****
	
	* First do dates *
	gen date_new = date(date, "YMD")
	format date_new %td
	drop date
	rename date_new date
	
	*Encode day of the week *
	encode day, gen(day_num)
	drop day
	rename day_num day
	
	* Indicator for if there are notes *
	gen abnormal_match = 0
	replace abnormal_match = 1 if notes != ""
	
	* Time (note that some matches do not have a start time)
	split time, p("(")
	split time1, p(":")
	rename time11 hour_start
	rename time12 minute_start
	drop time1 time2 time
	
	* Score *
	split score, p("–")
	rename score1 home_score
	rename score2 away_score
	drop score
	
	rename home home_team
	rename away away_team
	
	rename xg home_xg // note this is only there for France in recent years
	rename v9 away_xg
	
	
	* Destring variables *
	destring hour_start minute_start wk attendance home_score away_score season home_xg away_xg, replace
	
	gen goal_diff = home_score - away_score //clear HFA (sanity check)	
	gen goal_diff_xg = home_xg - away_xg //clear HFA (sanity check)	
	
	* Making everything look nice (order and label)
	order country season wk date day goal_diff goal_diff_xg home_team home_score home_xg away_team away_score away_xg referee venue attendance hour_start minute_start abnormal_match notes round
	
	la var country "Country of match"
	la var season "Season played - single number if inverted calendar"
	la var wk "Week of match"
	la var date "Date of match"
	la var day "Day of Week"
	la var goal_diff "Home - away goals"
	la var goal_diff_xg "Home - away xGoals"
	la var home_team "Home team (string)"
	la var home_score "Home team goals scored"
	la var home_xg "Home xGoals (only for recent years)"
	la var away_team "Away team (string)"
	la var away_score "Away team goals scored"
	la var away_xg "Away xGoals (only for recent years)"
	la var referee "Center ref for match"
	la var venue "Venue"
	la var attendance "Attendance"
	la var hour_start "Hour of match start"
	la var minute_start "Minute the match started (time is hour+minute)"
	la var abnormal_match "Indicator for if there is a note"
	la var notes "Notes associated with match"
	la var round "Round of match (regular season)"
	
	* Save space *
	compress
	
	save "${cleandata}/additional_leagues", replace

